/**
 * \file: mlink_dlt.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * MLINK GST Adapter
 *
 * \component: mlink
 *
 * \author: Michael Methner ADITG/SW1 mmethner@de.adit-jv.com
 *
 * \copyright: (c) 2003 - 2015 ADIT Corporation
 *
 * \history
 * 0.1 Michael Methner Initial version
 * 0.2 Bodo Winter Add dynamic audio switching feature
 *
 ***********************************************************************/

#include <mlink_gst.h>
#include <mlink_gst_internals.h>
#include <gst/gst.h>
#include <gst/base/gstbasesink.h>

#include <stdlib.h>
#include <gio/gio.h>

#define MAX_FADE_LENGTH_MS 500

/* PRQA: Lint Message 160, 505: deactivation because DLT macros generate this lint findings*/
/*lint -e160 -e505*/

/* PRQA: Lint Message 826: deactivation because casting mechanism of GObject throws the finding */
/*lint -e826*/

/* PRQA: Lint Message 751: deactivation because g_clear_pointer throws the finding */
/*lint -e751*/

/* PRQA: Lint Message 530: deactivation because va_list args gets initialized via va_start */
/*lint -e530*/

void mlink_gst_log_vargs(DltContext * ctx, DltLogLevelType level, char * log_string, ...)
{
   if(ctx)
   {
       char dltlog[DLT_USER_BUF_MAX_SIZE];
       va_list args;
       va_start(args,log_string);
       vsnprintf(dltlog, DLT_USER_BUF_MAX_SIZE, log_string, args);
       va_end(args);
       DLT_LOG(*ctx, level, DLT_STRING(dltlog));
   }
}
void mlink_gst_log(DltContext * ctx,DltLogLevelType level,char *msg)
{
    if (ctx)
    {
    	DLT_LOG(*ctx,level,DLT_STRING(msg));
    }
}
void mlink_gst_init(int *argc, char **argv[])
{
    gst_init(argc, argv);
}

int mlink_gst_parse_url(const char * url, unsigned int * ipaddr, unsigned int * port)
{
    char *ptr;
    int value;
    int i = 0;
    const char * tok = url;
    unsigned int url_len = strlen(url);

    if (0 != strncmp(url, "RTP://", 6) && 7 > url_len)
    {
        return 0;
    }

    tok = url + 5;

    do
    {
        tok++;
        value = strtol(tok, &ptr, 10);
        if (value < 0 || value > 255 || ptr == tok)
        {
            break;
        }

        *ipaddr = (*ipaddr << 8) + (unsigned int) value;
        i++;
        tok = strstr(tok, ".");
    }
    while (tok && ((unsigned int) (tok - url) < (url_len - 1)));

    if (i != 4)
    {
        return 0;
    }

    tok = strstr(url, ":");
    if (!tok || (unsigned int) (tok - url) > (1 + url_len))
    {
        return 0;
    }
    tok++;
    tok = strstr(tok, ":");
    if (!tok || (unsigned int) (tok - url) > 1 + url_len)
    {
        return 0;
    }
    tok++;

    value = strtol(tok, &ptr, 10);
    if (tok == ptr)
    {
        return 0;
    }
    *port = (unsigned int) value;

    return 1;
}

GstBusSyncReply mlink_gst_rtp_out_bus_call(GstBus *bus, GstMessage *msg, gpointer data)
{
    bus = bus;
    mlink_rtp_out_context * p_ctx = (mlink_rtp_out_context *) data;

    switch (GST_MESSAGE_TYPE(msg))
    {
        case GST_MESSAGE_APPLICATION:
        {
            g_main_loop_quit(p_ctx->gst_mainloop);
            break;
        }
        case GST_MESSAGE_ERROR:
        {
            gchar *debug;
            GError *error;
            gchar *err_info;
            gst_message_parse_error(msg, &error, &debug);
            if (debug)
            {
                err_info = g_strdup_printf("%s Debug Info: %s", error->message, debug);
            }
            else
            {
                err_info = g_strdup_printf("%s Debug Info: %s", error->message, "None");
            }
            mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,err_info);
            if (p_ctx->error_cb)
            {
                (p_ctx->error_cb)(p_ctx->p_user_ctx, p_ctx, err_info);
            }
            g_free(debug);
            g_free(err_info);
            g_error_free(error);

            break;
        }
        case GST_MESSAGE_STREAM_STATUS:
        {
            if (p_ctx->streaming_thread_cb)
            {
                GstStreamStatusType type;
                GstElement *owner = NULL;
                gst_message_parse_stream_status(msg, &type, &owner);
                const GValue * val = gst_message_get_stream_status_object(msg);

                if (val && GST_STREAM_STATUS_TYPE_ENTER == type)
                {
                    p_ctx->streaming_thread_cb(p_ctx->p_user_ctx, GST_ELEMENT_NAME(owner));
                }
            }
            break;
        }
        default:
            break;
    }
    return GST_BUS_PASS;
}

void mlink_gst_run_rtp_out_on_pad_added(GstElement *element, GstPad *pad,
        gpointer data)
{
    MLINK_UNUSED(element);
    GstPad *sinkpad;
    GstElement * rtpdepay = (GstElement *) data;

    sinkpad = gst_element_get_static_pad(rtpdepay, "sink");
    gst_pad_link(pad, sinkpad);
    g_clear_pointer (&sinkpad, gst_object_unref);
}


mlink_rtp_out_context * mlink_gst_start_rtp_out_streaming(const char * url,
        char * playback_device, unsigned int payloadtype, unsigned int ipl,
        mlink_gst_rtp_out_error_callback * error_cb,
        mlink_gst_rtp_out_data_callback * data_cb,
        mlink_gst_streaming_thread_callback * streaming_thread_cb,
        void * p_user_ctx, DltContext * p_dlt_ctx)
{
	mlink_gst_log(p_dlt_ctx,DLT_LOG_INFO,COMP_GIT_VERSION);
    if (!url || !playback_device || !error_cb)
    {
        mlink_gst_log(p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: invalid parameter");
        return NULL;
    }

    mlink_rtp_out_context * p_ctx = g_new0(mlink_rtp_out_context, 1);
    if (!p_ctx)
    {
        mlink_gst_log(p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not allocate context");
        return NULL;
    }
    p_ctx->p_dlt_ctx = p_dlt_ctx;
    p_ctx->playback_device = g_strdup((const gchar *)playback_device);
    if (!p_ctx->playback_device)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not allocate string for playback device");
        g_free(p_ctx);
        return NULL;
    }

    p_ctx->error_cb = error_cb;
    p_ctx->data_cb = data_cb;
    p_ctx->streaming_thread_cb = streaming_thread_cb;
    p_ctx->p_user_ctx = p_user_ctx;
    p_ctx->payloadtype = payloadtype;
    if (payloadtype == 100)
    {
        p_ctx->channels = 1;
        p_ctx->samplerate = 16000;
        p_ctx->ipl_ms = 1000 * ipl / 16000;
    }
    else if (payloadtype == 99)
    {
        p_ctx->channels = 2;
        p_ctx->samplerate = 48000;
        /* ipl is based on RTP payload type 99 (48kHz) [CCC-TS-012 Audio] */
        p_ctx->ipl_ms = 1000 * ipl / 48000;

    }
    else if (payloadtype == 98)
    {
        p_ctx->channels = 1;
        p_ctx->samplerate = 48000;
        /* ipl is based on RTP payload type 99 (48kHz) [CCC-TS-012 Audio] */
        p_ctx->ipl_ms = 1000 * ipl / 48000;
    }
    else
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: unsupported payload type");
        g_free(p_ctx->playback_device);
        g_free(p_ctx);
        return NULL;
    }

    if (!mlink_gst_parse_url(url, &p_ctx->ipaddr, &p_ctx->port))
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not parse url");
        g_free(p_ctx->playback_device);
        g_free(p_ctx);
        return NULL;
    }

    p_ctx->pipeline = gst_pipeline_new("rtp_out_pipeline");
    if (!p_ctx->pipeline)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not create rtp out pipeline");
        g_free(p_ctx->playback_device);
        g_free(p_ctx);
        return NULL;
    }

    p_ctx->sinkElement = NULL;
    p_ctx->volumeElement = NULL;
    p_ctx->volumeBufferProbeHandlerId = -1;
    p_ctx->nextPreSinkBufferTime = 0;
    p_ctx->disconnected = FALSE;
    p_ctx->isFirstRTPPacket =TRUE;

    p_ctx->init_finished = 0;
    g_cond_init(&p_ctx->init_cond);
    g_mutex_init(&p_ctx->mutex);

    p_ctx->gst_thread = g_thread_new("RTP_out_pipeline", mlink_gst_run_rtp_out_pipeline, p_ctx);

    /* wait for pipeline mainloop to initialize
     * gcond can have spurious wakeups
     * https://developer.gnome.org/glib/stable/glib-Threads.html#GCond
     */
    g_mutex_lock(&p_ctx->mutex);
    while (p_ctx->init_finished == 0)
    {
       g_cond_wait(&p_ctx->init_cond, &p_ctx->mutex);
    }
    g_mutex_unlock(&p_ctx->mutex);

    return p_ctx;
}

void mlink_gst_stop_rtp_out_streaming(mlink_rtp_out_context * p_ctx)
{
    if (p_ctx->pipeline)
    {
        /* message is sent delayed to avoid deadlock if
         * mlink_gst_start_rtp_out_streaming was immediately called before.
         * (SWGIII-8769)
         */
        GSource *idle_source;
        idle_source = g_idle_source_new();
        g_source_set_callback(idle_source, mlink_gst_stop_rtp_out_mainloop, p_ctx, NULL);
        g_source_attach(idle_source, p_ctx->g_main_context);
        g_clear_pointer (&idle_source, g_source_unref);
    }

    g_thread_join(p_ctx->gst_thread);
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: rtp out streaming stopped");
    g_free(p_ctx->playback_device);
    g_free(p_ctx);
}

/////////////////////////////////////////////////////////////////////
//
// RTP IN
//
/////////////////////////////////////////////////////////////////////

GstBusSyncReply mlink_gst_rtp_in_bus_call(GstBus *bus, GstMessage *msg, gpointer data)
{
    bus = bus;
    mlink_rtp_in_context * p_ctx = (mlink_rtp_in_context *) data;

    switch (GST_MESSAGE_TYPE(msg))
    {
        case GST_MESSAGE_APPLICATION:
        {
            g_main_loop_quit(p_ctx->gst_mainloop);
            break;
        }
        case GST_MESSAGE_ERROR:
        {
            gchar *debug;
            GError *error;
            gchar *err_info;
            gst_message_parse_error(msg, &error, &debug);
            if (debug)
            {
                err_info = g_strdup_printf("%s Debug Info: %s", error->message, debug);
            }
            else
            {
                err_info = g_strdup_printf("%s Debug Info: %s", error->message, "None");
            }
            mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,err_info);
            if (p_ctx->error_cb)
            {
                (p_ctx->error_cb)(p_ctx->p_user_ctx, p_ctx, err_info);
            }
            g_free(debug);
            g_free(err_info);
            g_error_free(error);

            break;
        }
        case GST_MESSAGE_STREAM_STATUS:
        {
            if (p_ctx->streaming_thread_cb)
            {
                GstStreamStatusType type;
                GstElement *owner = NULL;
                gst_message_parse_stream_status(msg, &type, &owner);
                const GValue * val = gst_message_get_stream_status_object(msg);

                if (val && GST_STREAM_STATUS_TYPE_ENTER == type)
                {
                    p_ctx->streaming_thread_cb(p_ctx->p_user_ctx,
                            GST_ELEMENT_NAME(owner));
                }
            }
            break;
        }
        default:
            break;
    }
    return GST_BUS_PASS;
}



mlink_rtp_in_context * mlink_gst_start_rtp_in_streaming(const char * url,
        char * capture_device, unsigned int payloadtype,
        mlink_gst_rtp_in_error_callback * error_cb,
        mlink_gst_streaming_thread_callback * streaming_thread_cb,
        void * p_user_ctx, DltContext * p_dlt_ctx)
{
    mlink_gst_log(p_dlt_ctx,DLT_LOG_INFO,COMP_GIT_VERSION);
    mlink_rtp_in_context * p_ctx = g_new0(mlink_rtp_in_context, 1);
    if (!p_ctx)
    {
        return NULL;
    }
    p_ctx->p_dlt_ctx = p_dlt_ctx;

    p_ctx->capture_device = g_strdup((const gchar *)capture_device);
    if (!p_ctx->capture_device)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not allocate string for capture device");
        g_free(p_ctx);
        return NULL;
    }

    p_ctx->error_cb = error_cb;
    p_ctx->streaming_thread_cb = streaming_thread_cb;
    p_ctx->p_user_ctx = p_user_ctx;
    p_ctx->payloadtype = payloadtype;
    if (payloadtype == 100)
    {
        p_ctx->channels = 1;
        p_ctx->samplerate = 16000;
    }
    else if (payloadtype == 99)
    {
        p_ctx->channels = 2;
        p_ctx->samplerate = 48000;
    }
    else if (payloadtype == 98)
    {
        p_ctx->channels = 1;
        p_ctx->samplerate = 48000;
    }
    else
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: unsupported payloadtype for rtp in");
        g_free(p_ctx->capture_device);
        g_free(p_ctx);
        return NULL;
    }

    if (!mlink_gst_parse_url(url, &p_ctx->ipaddr, &p_ctx->port))
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not parse rtp in url");
        g_free(p_ctx->capture_device);
        g_free(p_ctx);
        return NULL;
    }

    p_ctx->pipeline = gst_pipeline_new("rtp_in_pipeline");
    if (!p_ctx->pipeline)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not create pipeline");
        g_free(p_ctx->capture_device);
        g_free(p_ctx);
        return NULL;

    }

    p_ctx->init_finished = 0;
    g_cond_init(&p_ctx->init_cond);
    g_mutex_init(&p_ctx->mutex);

    p_ctx->gst_thread = g_thread_new("RTP_in_pipeline", mlink_gst_run_rtp_in_pipeline, p_ctx);

    /* wait for pipeline mainloop to initialize
     * gcond can have spurious wakeups
     * https://developer.gnome.org/glib/stable/glib-Threads.html#GCond
     */
    g_mutex_lock(&p_ctx->mutex);
    while (p_ctx->init_finished == 0)
    {
       g_cond_wait(&p_ctx->init_cond, &p_ctx->mutex);
    }
    g_mutex_unlock(&p_ctx->mutex);

    return p_ctx;
}

void mlink_gst_stop_rtp_in_streaming(mlink_rtp_in_context * p_ctx)
{
    if (p_ctx->pipeline)
    {
        /* SOPL-2942: Port patch SWGIII-8769 for RTP-IN streaming.
         *
         * Sending of shutdown message is delayed until gst_mainloop starts
         * running, to avoid race condition, if mlink_gst_stop_rtp_in_streaming
         * is  immediately called after mlink_gst_start_rtp_in_streaming.
         *
         * This will ensure that mlink_gst_stop_rtp_in_mainloop() will not be
         * triggered until and unless g_main_loop_run(p_ctx->gst_mainloop) in
         * mlink_gst_start_rtp_in_streaming() is executed.
         */
        GSource *idle_source;
        idle_source = g_idle_source_new();
        g_source_set_callback(idle_source, mlink_gst_stop_rtp_in_mainloop, p_ctx, NULL);
        g_source_attach(idle_source, p_ctx->g_main_context);
        g_clear_pointer (&idle_source, g_source_unref);
    }

    g_thread_join(p_ctx->gst_thread);
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: rtp in streaming stopped");
    g_free(p_ctx->capture_device);
    g_free(p_ctx);
}

static void do_disconnect_sink(mlink_rtp_out_context * p_ctx)
{
    g_object_set(G_OBJECT(p_ctx->outputSelector), "active-pad", p_ctx->outputSelectorPad2, NULL);
    g_object_set(G_OBJECT(p_ctx->valveElement), "drop", TRUE, NULL);
    GstPad *sinkPad = gst_element_get_static_pad(p_ctx->sinkElement, "sink");
    GstPad *valvePad = gst_element_get_static_pad(p_ctx->valveElement, "src");
    if (!sinkPad || !valvePad || !gst_pad_unlink(valvePad, sinkPad))
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_WARN,"mlink_gst: unlink failed while disconnecting");
    }
    g_clear_pointer (&sinkPad, gst_object_unref);
    g_clear_pointer (&valvePad, gst_object_unref);

    gst_element_set_state(p_ctx->sinkElement, GST_STATE_NULL);
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: switching sink done");
}

static void do_connect_sink(mlink_rtp_out_context * p_ctx, gchar * deviceName)
{
    g_object_set(G_OBJECT(p_ctx->sinkElement), "device", deviceName, NULL);
    GstStateChangeReturn result = gst_element_set_state(p_ctx->sinkElement, GST_STATE_PAUSED);
    if (result == GST_STATE_CHANGE_ASYNC)
    {
        result = gst_element_get_state(p_ctx->sinkElement, NULL, NULL, GST_CLOCK_TIME_NONE);
    }
    gst_element_set_state(p_ctx->sinkElement, GST_STATE_PLAYING);

    GstPad *sinkPad = gst_element_get_static_pad(p_ctx->sinkElement, "sink");
    GstPad *valvePad = gst_element_get_static_pad(p_ctx->valveElement, "src");
    if (!sinkPad || !valvePad || gst_pad_link(valvePad, sinkPad) != GST_PAD_LINK_OK)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_WARN,"mlink_gst: link failed while connecting");
    }
    g_clear_pointer (&sinkPad, gst_object_unref);
    g_clear_pointer (&valvePad, gst_object_unref);

    g_object_set(G_OBJECT(p_ctx->valveElement), "drop", FALSE, NULL);
    g_object_set(G_OBJECT(p_ctx->outputSelector), "active-pad", p_ctx->outputSelectorPad1, NULL);
}

void mlink_gst_disconnect_audio_device(mlink_rtp_out_context * p_ctx, guint32 fade_out_ms,
        mlink_gst_ramp_type ramp_type)
{
    if (p_ctx == NULL)
    {
        return;
    }
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: Starting to disconnect sink");
    if (p_ctx->disconnected)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: Already disconnected!");
        return;
    }

    if (fade_out_ms > MAX_FADE_LENGTH_MS)
        fade_out_ms = MAX_FADE_LENGTH_MS;

    g_mutex_lock(&p_ctx->mutex);
    if (!p_ctx->pipeline)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_WARN,"mlink_gst: Pipeline is NULL!");
        g_mutex_unlock(&p_ctx->mutex);
        return;
    }

    if (fade_out_ms > 0)
    {
        GstClockTime currentTime = p_ctx->nextPreSinkBufferTime;
        GstClockTime endTime = currentTime + fade_out_ms * GST_MSECOND;
        schedule_fade(p_ctx->volumeElement, currentTime, 1.0, 0.0, fade_out_ms, ramp_type,
                p_ctx->p_dlt_ctx);
        DLT_LOG(*p_ctx->p_dlt_ctx, DLT_LOG_DEBUG, DLT_STRING("mlink_gst: Start sleeping from"),
                DLT_FLOAT64((double) currentTime / GST_SECOND), DLT_STRING("until"),
                DLT_FLOAT64(((double) endTime / GST_SECOND)));

        GstBaseSink *baseSink = GST_BASE_SINK(p_ctx->sinkElement);
#ifdef GST_VERSION_1
        GST_BASE_SINK_PREROLL_LOCK(baseSink);
        gst_base_sink_wait_clock(baseSink, endTime, NULL);
        GST_BASE_SINK_PREROLL_UNLOCK(baseSink);
#else
        GstPad *pad = GST_BASE_SINK_PAD(baseSink);
        g_mutex_lock(pad->preroll_lock);
        gst_base_sink_wait_clock(baseSink, endTime, NULL);
        g_mutex_unlock(pad->preroll_lock);
#endif
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: Waiting done");
    }

    do_disconnect_sink(p_ctx);
    p_ctx->disconnected = TRUE;
    g_mutex_unlock(&p_ctx->mutex);
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: Disconnecting sink done");
}

void mlink_gst_connect_audio_device(mlink_rtp_out_context * p_ctx, gchar * device_name,
        guint32 fade_in_ms, mlink_gst_ramp_type ramp_type)
{
    if (p_ctx == NULL)
    {
        return;
    }
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: Starting to connect sink");
    if (!p_ctx->disconnected)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: Already connected!");
        return;
    }

    if (fade_in_ms > MAX_FADE_LENGTH_MS)
        fade_in_ms = MAX_FADE_LENGTH_MS;

    g_mutex_lock(&p_ctx->mutex);
    if (!p_ctx->pipeline)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_WARN,"mlink_gst: Pipeline is NULL!");
        g_mutex_unlock(&p_ctx->mutex);
        return;
    }
    DLT_LOG(*p_ctx->p_dlt_ctx, DLT_LOG_DEBUG, DLT_STRING("mlink_gst: Using device"),
            DLT_STRING(device_name));

    do_connect_sink(p_ctx, device_name);
    p_ctx->disconnected = FALSE;

    p_ctx->marker_bit_observed = FALSE;
    p_ctx->new_data_after_marker_bit_observed = FALSE;

    if (fade_in_ms > 0)
    {
        GstClock *pipeClock = gst_element_get_clock(p_ctx->pipeline);
        GstClockTime currentClockTime = gst_clock_get_time(pipeClock);
        GstClockTime volBaseTime = gst_element_get_base_time(p_ctx->volumeElement);
        GstClockTime volTime = currentClockTime - volBaseTime;
        schedule_fade(p_ctx->volumeElement, volTime, 0.0, 1.0, fade_in_ms, ramp_type,
                p_ctx->p_dlt_ctx);
    }

    g_mutex_unlock(&p_ctx->mutex);
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: Connecting sink done");
}

/*lint +e530*/
/*lint +e751*/
/*lint +e826*/
/*lint +e160 +e505*/
